﻿namespace Microsoft.Samples.PlanMyNight.Bing
{
    using System;
    using System.Collections.Specialized;
    using System.Configuration;
    using System.Globalization;
    using System.Linq;
    using System.Net;
    using System.Text;
    using Microsoft.Samples.PlanMyNight.Bing.VESearchService;
    using Microsoft.Samples.PlanMyNight.Entities;

    public class BingMapsService : IBingMapsService
    {
        public string GetClientToken(string clientIPAddress, int tokenValidityDurationMinutes)
        {
            // Create a reference to the web service (VEStagingToken.CommonService)
            using (var commonService = new Microsoft.Samples.PlanMyNight.Bing.VEStagingToken.CommonService())
            {
            commonService.Url =
              "https://staging.common.virtualearth.net/find-30/common.asmx";

            // Use your developer credentials
            NameValueCollection appSettings = ConfigurationManager.AppSettings;
            if (appSettings == null)
            {
                throw new ConfigurationErrorsException("Error: BingMapsService: Failed to load the application settings.");
            }

            commonService.Credentials = new NetworkCredential(appSettings["bingMapsAccountId"], appSettings["bingMapsAccountPassword"]);

            // Create the TokenSpecification object to pass to GetClientToken.
            var tokenSpec = new Microsoft.Samples.PlanMyNight.Bing.VEStagingToken.TokenSpecification();

            tokenSpec.ClientIPAddress = clientIPAddress;

            // The maximum allowable token duration is 480 minutes (8 hours).
            // The minimum allowable duration is 15 minutes.
            tokenSpec.TokenValidityDurationMinutes = tokenValidityDurationMinutes;

            try
            {
                return commonService.GetClientToken(tokenSpec);
            }
            catch (WebException)
            {
                return null;
            }
        }
        }

        public BingCoordinate GeocodeAddress(ActivityAddress address, string token)
        {
            var geocodeRequest = new Microsoft.Samples.PlanMyNight.Bing.VEGeocodingService.GeocodeRequest();

            // Set the credentials using a valid Bing Maps token
            geocodeRequest.Credentials = new Microsoft.Samples.PlanMyNight.Bing.VEGeocodingService.Credentials();
            geocodeRequest.Credentials.Token = token;

            // Set the full address query
            geocodeRequest.Address = new Microsoft.Samples.PlanMyNight.Bing.VEGeocodingService.Address()
            {
                AddressLine = address.StreetAddress,
                Locality = address.City,
                PostalCode = address.Zip,
                AdminDistrict = address.State,
                CountryRegion = "US"
            };

            // Set the options to only return high confidence results 
            var filters = new Microsoft.Samples.PlanMyNight.Bing.VEGeocodingService.ConfidenceFilter[1];
            filters[0] = new Microsoft.Samples.PlanMyNight.Bing.VEGeocodingService.ConfidenceFilter();
            filters[0].MinimumConfidence = Microsoft.Samples.PlanMyNight.Bing.VEGeocodingService.Confidence.High;

            // Add the filters to the options
            var geocodeOptions = new Microsoft.Samples.PlanMyNight.Bing.VEGeocodingService.GeocodeOptions();
            geocodeOptions.Filters = filters;
            geocodeRequest.Options = geocodeOptions;

            Microsoft.Samples.PlanMyNight.Bing.VEGeocodingService.GeocodeResponse geocodeResponse = null;

            // Make the geocode request
            using (var geocodeService = new Microsoft.Samples.PlanMyNight.Bing.VEGeocodingService.GeocodeServiceClient())
            {
                try
                {
                    geocodeResponse = geocodeService.Geocode(geocodeRequest);
                    geocodeService.Close();
                }
                catch
                {
                    geocodeService.Abort();
                }
            }

            if (geocodeResponse != null && geocodeResponse.Results != null && geocodeResponse.Results.Length > 0)
            {
                var location = geocodeResponse.Results[0].Locations[0];
                return new BingCoordinate { Latitude = (float)location.Latitude, Longitude = (float)location.Longitude };
            }

            return default(BingCoordinate);
        }

        public PagingResult<Activity> SearchActivities(AdvancedSearchQuery searchCriteria, string token)
        {
            // request
            SearchRequest searchRequest = GetBaseSearchRequest(token, searchCriteria.PageSize, searchCriteria.Page);

            // category filter - trick bing maps to use the same category twice
            var filter = new FilterExpressionClause();
            filter.Expressions = new FilterExpressionBase[]
                                     {
                                         new FilterExpression()
                                             {
                                                 CompareOperator = CompareOperator.Equals,
                                                 PropertyId = 1,            // Category
                                                 FilterValue = searchCriteria.ActivityTypeId
                                             },
                                         new FilterExpression()
                                             {
                                                 CompareOperator = CompareOperator.Equals,
                                                 PropertyId = 1,            // Category
                                                 FilterValue = searchCriteria.ActivityTypeId
                                             },
                                     };

            searchRequest.SearchOptions.Filters = filter;
            searchRequest.SearchOptions.SortOrder = (SortOrder)Enum.Parse(typeof(SortOrder), searchCriteria.SortBy.ToString(), true);

            // location options
            if (searchCriteria.Longitude.HasValue && searchCriteria.Latitude.HasValue)
            {
                // using coordinates
                searchRequest.StructuredQuery = new StructuredSearchQuery
                {
                    Location = string.Format(CultureInfo.InvariantCulture, "{0}, {1}", searchCriteria.Latitude.Value, searchCriteria.Longitude.Value)
                };
            }
            else
            {
                // using free text query
                searchRequest.Query = GetSearchQuery(searchCriteria);
            }

            // invoke
            return this.InvokeSearch(searchRequest, searchCriteria.PageSize, searchCriteria.Page).Item1;
        }

        public Tuple<PagingResult<Activity>, ActivityAddress> SearchActivities(NaturalSearchQuery query, string token)
        {
            // request
            SearchRequest searchRequest = GetBaseSearchRequest(token, query.PageSize, query.Page);

            // category filter - trick bing maps to use the same category twice
            var filter = new FilterExpressionClause();
            filter.Expressions = new FilterExpressionBase[]
                                     {
                                         new FilterExpression()
                                             {
                                                 CompareOperator = CompareOperator.Equals,
                                                 PropertyId = 1,            // Category
                                                 FilterValue = query.ActivityTypeId
                                             },
                                         new FilterExpression()
                                             {
                                                 CompareOperator = CompareOperator.Equals,
                                                 PropertyId = 1,            // Category
                                                 FilterValue = query.ActivityTypeId
                                             },
                                     };

            searchRequest.SearchOptions.Filters = filter;
            searchRequest.SearchOptions.SortOrder = (SortOrder)Enum.Parse(typeof(SortOrder), query.SortBy.ToString(), true);
            searchRequest.Query = query.Query;

            return this.InvokeSearch(searchRequest, query.PageSize, query.Page);
        }

        protected virtual Tuple<PagingResult<Activity>, ActivityAddress> InvokeSearch(Microsoft.Samples.PlanMyNight.Bing.VESearchService.SearchRequest searchRequest, int pageSize, int page)
        {
            using (var client = new SearchServiceClient())
            {
            try
            {
                client.Open();
                var searchResponse = client.Search(searchRequest);

                if (searchResponse.ResultSets.Length == 0)
                {
                    // nothing was found
                    return new Tuple<PagingResult<Activity>, ActivityAddress>(
                        new PagingResult<Activity>(new Activity[0])
                        {
                            TotalItems = 0,
                            CurrentPage = page,
                            PageSize = pageSize
                        },
                        null);
                }

                var resultSet = searchResponse.ResultSets[0];

                var items = resultSet.Results.Select(r => r as BusinessSearchResult)
                                             .Select(br => new Activity
                                                    {
                                                        BingId = br.Id,
                                                        Name = br.Name,
                                                        Street = br.Address.AddressLine,
                                                        State = br.Address.AdminDistrict,
                                                        City = br.Address.Locality,
                                                        Zip = br.Address.PostalCode ?? string.Empty,
                                                        PhoneNumber = br.PhoneNumber,
                                                        Location = new[] { br.LocationData.Locations[0].Longitude, br.LocationData.Locations[0].Latitude },
                                                        Distance = br.Distance,
                                                        Rating = br.UserRating,
                                                        RatingCount = br.RatingCount
                                                    });

                var results = new PagingResult<Activity>(items)
                {
                    CurrentPage = page,
                    PageSize = pageSize,
                    TotalItems = resultSet.EstimatedMatches,
                };

                ActivityAddress parsedLocation = null;
                if (searchResponse.ResultSets[0].SearchRegion != null)
                {
                    parsedLocation = new ActivityAddress
                    {
                        City = searchResponse.ResultSets[0].SearchRegion.GeocodeLocation.Address.Locality,
                        State = searchResponse.ResultSets[0].SearchRegion.GeocodeLocation.Address.AdminDistrict
                    };
                }

                return new Tuple<PagingResult<Activity>, ActivityAddress>(results, parsedLocation);
            }
            finally
            {
                if (client != null)
                {
                    client.Close();
                }
            }
        }
        }

        private static Microsoft.Samples.PlanMyNight.Bing.VESearchService.SearchRequest GetBaseSearchRequest(string token, int pageSize, int page)
        {
            SearchRequest searchRequest = new SearchRequest();
            searchRequest.Credentials = new Microsoft.Samples.PlanMyNight.Bing.VESearchService.Credentials();
            searchRequest.Credentials.Token = token;
            searchRequest.SearchOptions = new SearchOptions();
            searchRequest.SearchOptions.Count = pageSize;
            searchRequest.SearchOptions.StartingIndex = (page - 1) * pageSize;
            searchRequest.SearchOptions.ListingType = ListingType.Business;
            searchRequest.UserProfile = new Microsoft.Samples.PlanMyNight.Bing.VESearchService.UserProfile
            {
                DistanceUnit = DistanceUnit.Mile,
            };

            return searchRequest;
        }

        private static string GetSearchQuery(AdvancedSearchQuery searchCriteria)
        {
            var delimiter = ", ";
            var query = new StringBuilder();

            if (!string.IsNullOrEmpty(searchCriteria.StreetAddress))
            {
                query.Append(searchCriteria.StreetAddress);
                query.Append(delimiter);
            }

            if (!string.IsNullOrEmpty(searchCriteria.City))
            {
                query.Append(searchCriteria.City);
                query.Append(delimiter);
            }

            if (!string.IsNullOrEmpty(searchCriteria.Zip))
            {
                query.Append(searchCriteria.Zip);
                query.Append(delimiter);
            }

            if (!string.IsNullOrEmpty(searchCriteria.State))
            {
                query.Append(searchCriteria.State);
                query.Append(delimiter);
            }

            return query.ToString();
        }
    }
}
